function demo_RBF_CE(dataset, seqName, varargin)
	% K-class classification with Georeg under RBF-based representation
	%
	% based on Qinxun Bai's implementation for the following paper:
    % Qinxun Bai, Steven Rosenberg, Zheng Wu, and Stan Sclaroff.
    % "Differential Geometric Regularization for Supervised Learning of
    % Classifiers". ICML 2016
    %
	% modified by Kun He, June 2016
	% modified by Qinxun Bai, Nov 2016

	if nargin < 1, dataset = 'UCI'; end
	if nargin < 2, seqName = 'liver'; end

	%clear;
	opts = get_opts(dataset, seqName, varargin{:});

	% load orig data
	[vec_Label, mat_Instance] = loadData(dataset, seqName); %[nTrain x 1, nTrain x Nd]
	whos vec_Label mat_Instance

	opts.Nd = size(mat_Instance, 2);
	opts.K  = max(vec_Label);  % # classes
	disp(opts)

	% n-folds cross-validation test
	split_file = sprintf('%s%s_%s_%dfolds.mat', ...
		opts.localdir, opts.dataset, opts.seqName, opts.cvfolds);

	cvData = CVsplit(mat_Instance, vec_Label, opts.cvfolds, 1, split_file);
	cvErr  = zeros(1, opts.cvfolds);

	for ifold = 1 : opts.cvfolds
		Xtrain = cvData.train{ifold};
		Xtest  = cvData.test {ifold};
		Ytrain = cvData.train_label{ifold};
		Ytest  = cvData.test_label {ifold};

		% rescale to [0, scale]
        for i = 1 : opts.Nd
			mat_min = min(Xtrain(:,i));
			mat_max = max(Xtrain(:,i));
			Xtrain(:,i) = opts.scale*(Xtrain(:,i)-mat_min)/(mat_max-mat_min);
			Xtest(:,i) = opts.scale*(Xtest(:,i)-mat_min)/(mat_max-mat_min);
        end
                
        % find the class that has least samples
        siz_class = arrayfun(@(x) sum(Ytrain==x), 1:opts.K);
        [leastK, opts.id_leastK] = min(siz_class);

		% cross-validation on sigma/gamma using train folds
		myLogInfo('Fold %d/%d: findpara_cv(%d) on Xtrain...', ifold, opts.cvfolds, opts.parafolds); 
		t1 = tic;
		[sigma, gamma, cv_err] = findpara_cv(Xtrain, Ytrain, opts);
		toc(t1);
		myLogInfo('Fold %d/%d: sigma = %g, gamma = %g, train_err = %.2f', ...
			ifold, opts.cvfolds, sigma, gamma, 100*cv_err);

		% train
		model = train_RBF_CE(Xtrain, Ytrain, sigma, gamma, opts, [], 1);

		% test
		cvErr(ifold) = test_RBF(Xtest, Ytest, Xtrain, Ytrain, model, opts);
		myLogInfo('Fold %d/%d: [test_err = %.2f]\n', ...
			ifold, opts.cvfolds, cvErr(ifold)*100);

	end

	myLogInfo('%s_%s: %d-CV err = %g', dataset, seqName, opts.cvfolds, 100*mean(cvErr));

end


% --------------------------------------------------------------
function [sigma, gamma, minerr] = findpara_cv(X, Y, opts)

	cvData  = CVsplit(X, Y, opts.parafolds, 0);
	paraErr = zeros(length(opts.sigmas), length(opts.gammas));

	for is = 1 : length(opts.sigmas)
		sigma_i = opts.sigmas(is);

		% compute RBF representation for fixed sigma_i
		RBFs = [];
		RBFs(opts.parafolds).inv_H = [];
		for iCV = 1:opts.parafolds
			[H, RBFs(iCV).H_deriv] = RBF(cvData.train{iCV}, sigma_i);
			I = eye(size(cvData.train{iCV}, 1));
			RBFs(iCV).inv_H = (H + opts.lambda*I)\I;
		end

		for ig = 1 : length(opts.gammas)
			gamma_i = opts.gammas(ig);

			% do CV on (sigma_i, gamma_i)
			cvErr = zeros(1, opts.parafolds);
			for iCV = 1 : opts.parafolds
				cvXtrain = cvData.train{iCV};
				cvXtest  = cvData.test{iCV};
				cvYtrain = cvData.train_label{iCV};
				cvYtest  = cvData.test_label{iCV};

				model = train_RBF_CE(cvXtrain, cvYtrain, sigma_i, gamma_i, opts, RBFs(iCV));
				cvErr(iCV) = test_RBF(cvXtest, cvYtest, cvXtrain, cvYtrain, model, opts);
			end

			paraErr(is, ig) = mean(cvErr);
		end
	end

	[minerr, idx] = min(paraErr(:));
	[is, ig] = ind2sub(size(paraErr), idx);
	sigma = opts.sigmas(is);
	gamma = opts.gammas(ig);
end


% --------------------------------------------------------------
function err = test_RBF(Xtest, Ytest, Xtrain, Ytrain, model, opts)
	nTest = length(Ytest);
	score = f_RBF(Xtest, Xtrain, model.sigma) * model.alpha;
	%{
	score = zeros(nTest, opts.K);
	for i = 1 : nTest
		score(i, :) = f_RBF(Xtest(i, :), Xtrain, model.sigma) * model.alpha;
	end
	%}
	[~, Ypred] = max(softmax(score), [], 2);                                    
	err = mean(Ytest ~= Ypred);
end


% --------------------------------------------------------------
function model = train_RBF_CE(Xtrain, Ytrain, sigma, gamma, opts, RBFs, verb)
	if nargin < 6, RBFs = []; end
	if nargin < 7, verb = 0; end

	ntrain = length(Ytrain);
	Sigma  = sigma * ones(1, ntrain);                

	% initialize RBFs
	if isempty(RBFs)
		t0 = tic;
		[H, H_deriv] = RBF(Xtrain, Sigma);
		inv_H = ( H + opts.lambda*eye(ntrain) ) \ eye(ntrain);                
		if verb, myLogInfo('%d RBFs init''d in %.2f sec', ntrain, toc(t0)); end
	else
		inv_H = RBFs.inv_H;
		H_deriv = RBFs.H_deriv;
    end        
	  
	F  = ones(ntrain, opts.K);
	Y  = ones(ntrain, opts.K)/opts.K;
	t0 = tic;
	if verb, myLogInfo('Starting GF, sigma = %g, gamma = %g', sigma, gamma); end

	% zero step
	grad1 = gradDist_RBF_CE(Ytrain, Y);
	F     = F - opts.sz(1)*grad1;                
	alpha = inv_H * F;  % ntrain x K

	% following steps 
	for iter = 1 : opts.niter
		Y     = softmax(F);
		grad1 = gradDist_RBF_CE(Ytrain, Y);
        
        % for simplex
		grad2 = gradVol_RBF(Xtrain, alpha, Sigma, H_deriv, opts.id_leastK, Y); 
        
		F     = F - opts.sz(iter)*grad1 - opts.sz(iter)*gamma*grad2;
		alpha = inv_H * F;   
	end
	if verb
		myLogInfo('%d GF iters in %.2f sec', opts.niter, toc(t0));
	end
	model = [];
	model.alpha = alpha;
	model.sigma = sigma;
end


% --------------------------------------------------------------
function cv_data = CVsplit(mat_Instance, vec_Label, nfold, checkDup, split_file)
	try
		cv_data = load(split_file); 
		myLogInfo('Loaded %s', split_file);
		return;
	end

	% do n-fold split
	train       = cell(1, nfold);
	test        = cell(1, nfold);
	train_label = cell(1, nfold);
	test_label  = cell(1, nfold);
	for i = 1 : nfold
		test{i} = mat_Instance(i:nfold:end, :); % nTest x Nd
		test_label{i} = vec_Label(i:nfold:end); % nTest x 1
		train{i} = mat_Instance(setdiff(1:end, i:nfold:end), :); % nTrain x Nd
		train_label{i} = vec_Label(setdiff(1:end, i:nfold:end)); % nTrain x 1

		% delete duplicate samples in train
        if checkDup
            id_del = [];
            for k  = 1 : size(train{i},1)
                for j = 1 : k-1
%                     v = train(i,:) - train(j,:);
%                     if all(v==0)
                    if isequal(train{i}(k, :), train{i}(j, :))
                        id_del = [id_del, k];
                        break;
                    end
                end
            end
            train{i}(id_del,:) = [];
            train_label{i}(id_del,:) = [];
        end
	end

	if (nargin > 4) && ~isempty(split_file)
		save(split_file, 'train', 'test', 'train_label', 'test_label');
		myLogInfo('Saved to %s', split_file);
	end
	cv_data = [];
	cv_data.test        = test;
	cv_data.train       = train;
	cv_data.test_label  = test_label;
	cv_data.train_label = train_label;
end

